# SPDX-FileCopyrightText: 2017 Michael McWethy for Adafruit Industries
#
# SPDX-License-Identifier: MIT

"""
`adafruit_sh1106`
====================================================

CircuitPython SH1106 OLED driver, I2C and SPI interfaces

This library is based on the Adafruit_CircuitPython_SSD1306
driver written by: Tony DiCola, Michael McWethy

SSH1106 adaptation: István Cserny
"""

import time

from micropython import const
from adafruit_bus_device import i2c_device, spi_device

try:
    import framebuf
except ImportError:
    import adafruit_framebuf as framebuf

try:
    # Used only for typing
    from typing import Optional
    import busio
    import digitalio
except ImportError:
    pass

__version__ = "0.0.0-auto.0"
__repo__ = "https://github.com/adafruit/Adafruit_CircuitPython_SH1106.git"

# register definitions
SET_CONTRAST = const(0x81)
SET_ENTIRE_ON = const(0xA4)
SET_NORM_INV = const(0xA6)
SET_DISP = const(0xAE)
SET_MEM_ADDR = const(0x20)
SET_COL_ADDR = const(0x21)
SET_PAGE_ADDR = const(0x22)
SET_DISP_START_LINE = const(0x40)
SET_SEG_REMAP = const(0xA0)
SET_MUX_RATIO = const(0xA8)
SET_IREF_SELECT = const(0xAD)
SET_COM_OUT_DIR = const(0xC0)
SET_DISP_OFFSET = const(0xD3)
SET_COM_PIN_CFG = const(0xDA)
SET_DISP_CLK_DIV = const(0xD5)
SET_PRECHARGE = const(0xD9)
SET_VCOM_DESEL = const(0xDB)
SET_CHARGE_PUMP = const(0x8D)


class _SH1106(framebuf.FrameBuffer):
    """Base class for SH1106 display driver"""

    # pylint: disable-msg=too-many-arguments
    def __init__(
        self,
        buffer: memoryview,
        width: int,
        height: int,
        *,
        external_vcc: bool,
        reset: Optional[digitalio.DigitalInOut],
        page_addressing: bool
    ):
        super().__init__(buffer, width, height)
        self.width = width
        self.height = height
        self.external_vcc = external_vcc
        # reset may be None if not needed
        self.reset_pin = reset
        self.page_addressing = True      # SH1106 has only Page Addressing mode
        if self.reset_pin:
            self.reset_pin.switch_to_output(value=0)
        self.pages = self.height // 8
        # Note the subclass must initialize self.framebuf to a framebuffer.
        # This is necessary because the underlying data buffer is different
        # between I2C and SPI implementations (I2C needs an extra byte).
        self._power = False
        # Parameters for efficient Page Addressing Mode (typical of U8Glib libraries)
        # Important as not all screens appear to support Horizontal Addressing Mode
        self.pagebuffer = bytearray(width + 1)  # type: Optional[bytearray]
        self.pagebuffer[0] = 0x40  # Set first byte of data buffer to Co=0, D/C=1
        self.page_column_start = bytearray(2)  # type: Optional[bytearray]
        col_offset = (132 - self.width) // 2
        self.page_column_start[0] = col_offset % 32
        self.page_column_start[1] = 0x10 + col_offset // 32
        # Let's get moving!
        self.poweron()
        self.init_display()

    @property
    def power(self) -> bool:
        """True if the display is currently powered on, otherwise False"""
        return self._power

    def init_display(self) -> None:
        """Base class to initialize display"""
        for cmd in (
            0xAE,           #--display off
            0xA1,           #--set segment re-map 127 to 0
            0xC8,           #--Set COM Out Scan Dir 63 to 0
            0x32,           #--set pump voltage value to 8.0V (SH1106 only)
            0x40,           #--set start line address
            0x81,           #--set contrast control register
            0x80,           # POR value = 80
            0xA4,           #--set normal mode (A5: test mode)
            0xA6,           #--set normal display (i.e non-inverted mode)
            0xA8,           #--set multiplex ratio(1 to 64)
            0x3F,           #
            0xAD,           #--set DC-DC mode
            0x8B,           # DC-DC converter ON 
            0xD3,           #--set display offset
            0x00,           # no offset 
            0xD5,           #--set display clock divide ratio/oscillator frequency
            0x50,           # set frequency and divide ratio
            0xD9,           #--set dis-charge/pre-charge period
            0x22,           #
            0xDA,           #--set com pins hardware configuration
            0x12,
            0xDB,           #--set vcomh
            0x35,           # SH1106: 0x35, SSD1306: 0x20 0.77xVcc
            0xAF,           #--turn on SSD1306 panel
            ):
            self.write_cmd(cmd)
        self.fill(0)
        self.show()

    def poweroff(self) -> None:
        """Turn off the display (nothing visible)"""
        self.write_cmd(SET_DISP)
        self._power = False

    def contrast(self, contrast: int) -> None:
        """Adjust the contrast"""
        self.write_cmd(SET_CONTRAST)
        self.write_cmd(contrast)

    def invert(self, invert: bool) -> None:
        """Invert all pixels on the display"""
        self.write_cmd(SET_NORM_INV | (invert & 1))

    def rotate(self, rotate: bool) -> None:
        """Rotate the display 0 or 180 degrees"""
        self.write_cmd(SET_COM_OUT_DIR | ((rotate & 1) << 3))
        self.write_cmd(SET_SEG_REMAP | (rotate & 1))
        # com output (vertical mirror) is changed immediately
        # you need to call show() for the seg remap to be visible

    def write_framebuf(self) -> None:
        """Derived class must implement this"""
        raise NotImplementedError

    def write_cmd(self, cmd: int) -> None:
        """Derived class must implement this"""
        raise NotImplementedError

    def poweron(self) -> None:
        "Reset device and turn on the display."
        if self.reset_pin:
            self.reset_pin.value = 1
            time.sleep(0.001)
            self.reset_pin.value = 0
            time.sleep(0.010)
            self.reset_pin.value = 1
            time.sleep(0.010)
        self.write_cmd(SET_DISP | 0x01)
        self._power = True

    def show(self) -> None:
        """Update the display"""
        self.write_framebuf()


class SH1106_I2C(_SH1106):
    """
    I2C class for SH1106

    :param width: the width of the physical screen in pixels,
    :param height: the height of the physical screen in pixels,
    :param i2c: the I2C peripheral to use,
    :param addr: the 8-bit bus address of the device,
    :param external_vcc: whether external high-voltage source is connected.
    :param reset: if needed, DigitalInOut designating reset pin
    """

    def __init__(
        self,
        width: int,
        height: int,
        i2c: busio.I2C,
        *,
        addr: int = 0x3C,
        external_vcc: bool = False,
        reset: Optional[digitalio.DigitalInOut] = None,
        page_addressing: bool = True
    ):
        self.i2c_device = i2c_device.I2CDevice(i2c, addr)
        self.addr = addr
        self.page_addressing = page_addressing
        self.temp = bytearray(2)
        # Add an extra byte to the data buffer to hold an I2C data/command byte
        # to use hardware-compatible I2C transactions.  A memoryview of the
        # buffer is used to mask this byte from the framebuffer operations
        # (without a major memory hit as memoryview doesn't copy to a separate
        # buffer).
        self.buffer = bytearray(((height // 8) * width) + 1)
        self.buffer[0] = 0x40  # Set first byte of data buffer to Co=0, D/C=1
        super().__init__(
            memoryview(self.buffer)[1:],
            width,
            height,
            external_vcc=external_vcc,
            reset=reset,
            page_addressing=True,
        )

    def write_cmd(self, cmd: int) -> None:
        """Send a command to the I2C device"""
        self.temp[0] = 0x00  # Co=0, D/C#=0
        self.temp[1] = cmd
        with self.i2c_device:
            self.i2c_device.write(self.temp)

    def write_framebuf(self) -> None:
        """Blast out the frame buffer through the hardware I2C interfaces."""
        for page in range(self.pages):
            self.write_cmd(0xB0 + page)
            self.write_cmd(self.page_column_start[0])
            self.write_cmd(self.page_column_start[1])
            self.pagebuffer[1:] = self.buffer[
                1 + self.width * page : 1 + self.width * (page + 1)
            ]
            with self.i2c_device:
                self.i2c_device.write(self.pagebuffer)


