# SPDX-FileCopyrightText: 2021 ladyada for Adafruit Industries
# SPDX-License-Identifier: MIT
#
# Based on the Adafruit example: st7735r_128x160_simpletest.py
# Original code © Adafruit Industries.
#
# Modifications by I. Cserny, 2026:
# - Adapted for ESP32-C3 microcontroller
# - Changed display initialization to portrait orientation
#
# This test will initialize the display using displayio and draw a solid green
# background, a smaller purple rectangle, and some yellow text
#
# displayio object hierarchy for this example:
# # display.root_group
# └── splash (Group)
#     ├── bg_sprite (TileGrid)        # full-screen green background
#     ├── inner_sprite (TileGrid)     # inset purple rectangle
#     └── text_group (Group, scale=2) # subgroup used for text scaling
#         └── text_area (Label)       # "Hello World!" text
#

import board
import displayio
import terminalio
from adafruit_display_text import label
from fourwire import FourWire
from adafruit_st7735r import ST7735R

# Release any resources currently in use for the displays
displayio.release_displays()

spi = board.SPI()
tft_cs = board.IO7
tft_dc = board.IO3

display_bus = FourWire(spi, command=tft_dc, chip_select=tft_cs, reset=board.IO2)
display = ST7735R(display_bus, width=128, height=160, rotation=0, bgr=True)

# Make the display context
splash = displayio.Group()
display.root_group = splash

color_bitmap = displayio.Bitmap(128, 160, 1)
color_palette = displayio.Palette(1)
color_palette[0] = 0x00FF00  # Bright Green

bg_sprite = displayio.TileGrid(color_bitmap, pixel_shader=color_palette, x=0, y=0)
splash.append(bg_sprite)

# Draw a smaller inner rectangle
inner_bitmap = displayio.Bitmap(118, 150, 1)
inner_palette = displayio.Palette(1)
inner_palette[0] = 0xAA0088  # Purple
inner_sprite = displayio.TileGrid(inner_bitmap, pixel_shader=inner_palette, x=5, y=5)
splash.append(inner_sprite)

# Draw a text label ("Hello World!") scaled ×2
# Text is placed inside a subgroup so the entire label can be scaled uniformly.
# In displayio, Label objects themselves cannot be scaled directly; only Groups
# support scaling. Wrapping the Label in a Group(scale=2) enlarges the text
# without needing a larger font.
text_group = displayio.Group(scale=2, x=18, y=64)
text = "Hello \nWorld!"
text_area = label.Label(terminalio.FONT, text=text, color=0xFFFF00) #Yellow
text_group.append(text_area)  # Subgroup for text scaling
splash.append(text_group)

while True:
    pass
