"""
Blood Pressure Diagram Demo – displayio + shapes + bitmap
---------------------------------------------------------

Ez a program egy 2.9" háromszínű (fekete–fehér–piros) IL0373 e‑papír kijelzőn
mutat be egy egyszerű, de professzionális megjelenésű vérnyomásdiagramot.
A cél egy olyan demonstráció, amelyen keresztül megismerhető a displayio
hierarchiája, a bitmap betöltés, valamint a displayio shapes modul
(Line, Circle) használata.

A program fix, előre definiált adatokkal dolgozik (31 nap SYS/DIA értékei),
és egy háttér-bitmapre rajzolja rá a két görbét. A megoldás könnyen
továbbfejleszthető dinamikus adatokra, más grafikonokra vagy teljes
dashboardokra.

---------------------------------------------------------
Telepítendő CircuitPython könyvtárak (lib/ könyvtárba):
---------------------------------------------------------
• adafruit_il0373            – az IL0373 e‑papír driver
• adafruit_display_text      – szövegkiíráshoz (Label)
• adafruit_display_shapes    – vonalak és körök rajzolásához

---------------------------------------------------------
Szükséges projektfájlok:
---------------------------------------------------------
• bp_background.bmp        – a háttérkép (a gyökérkönyvtárba)

---------------------------------------------------------
A displayio hierarchiája a programban:
---------------------------------------------------------
display.root_group = main
    ├── TileGrid (háttér bitmap)
    ├── Label (címsor)
    └── Group (graph)
         ├── Line objektumok (SYS és DIA görbék)
         └── Circle objektumok (adatpontok)

---------------------------------------------------------
Főbb változók:
---------------------------------------------------------
W, H        – kijelző mérete pixelben
ML, MR,
MT, MB      – margók (bal, jobb, felső, alsó)
PW, PH      – hasznos rajzterület szélessége/magassága
X_MIN,
X_MAX       – X tengely skálája (napok)
Y_MIN,
Y_MAX       – Y tengely skálája (vérnyomásértékek)

---------------------------------------------------------
Főbb függvények:
---------------------------------------------------------
x_to_px(day)
    A nap (1..31) értékét vízszintes pixelpozícióra alakítja.

y_to_px(value)
    A vérnyomásértéket függőleges pixelpozícióra alakítja.

Mindkét függvény lineáris transzformációt végez a skálázott
rajzterületre (PW × PH).

---------------------------------------------------------
Megjegyzés:
---------------------------------------------------------
A program demonstrációs célú, statikus adatokkal dolgozik.
A displayio struktúra és a shapes modul használata azonban
könnyen adaptálható bármilyen grafikonhoz, dashboardhoz vagy
idősoros megjelenítéshez.
"""


import time
import board
import displayio
import fourwire
import adafruit_il0373
from adafruit_display_shapes.circle import Circle
from adafruit_display_shapes.line import Line
import terminalio
from adafruit_display_text import label

# --- MÉRET ÉS MARGÓK ---
W, H = 296, 128
ML, MR, MT, MB = 26, 14, 12, 22

# --- SKÁLÁK ---
X_MIN, X_MAX = 0, 32
Y_MIN, Y_MAX = 60, 160

# --- HASZNOS TERÜLET ---
PW, PH = W - ML - MR, H - MT - MB

# --- TRANSZFORMÁCIÓK ---
def x_to_px(day):
    return int(ML + (day - X_MIN) * PW / (X_MAX - X_MIN))

def y_to_px(val):
    return int(H - MB - (val - Y_MIN) * PH / (Y_MAX - Y_MIN))

# --- SYS ÉS DIA ÉRTÉKEK (31 nap) ---
sys_values = [
    132, 118, 140, 127, 136, 112, 144, 121,
    129, 138, 115, 134, 142, 119, 131, 125,
    137, 113, 128, 133, 141, 117, 130, 139,
    122, 135, 114, 143, 126, 120, 124
]

dia_values = [
    82, 74, 88, 79, 85, 71, 90, 76,
    81, 87, 72, 84, 89, 75, 83, 78,
    86, 70, 80, 82, 88, 73, 84, 89,
    77, 85, 71, 90, 79, 74, 81
]

# --- DISPLAY INITIALIZÁLÁS ---
displayio.release_displays()

spi = board.SPI()
epd_cs = board.IO3
epd_dc = board.IO2
epd_reset = board.IO1
epd_busy = board.IO0

display_bus = fourwire.FourWire(
    spi, command=epd_dc, chip_select=epd_cs, reset=epd_reset, baudrate=1000000
)

print("EPD creation")
time.sleep(1)

display = adafruit_il0373.IL0373(
    display_bus,
    width=296,
    height=128,
    rotation=270,
    busy_pin=epd_busy,
    highlight_color=0xFF0000,   # piros réteg
)

# --- FŐ GROUP ---
main = displayio.Group()

# --- HÁTTÉRKÉP BETÖLTÉSE ---
bg_bitmap = displayio.OnDiskBitmap("/bp_background.bmp")
bg = displayio.TileGrid(bg_bitmap, pixel_shader=bg_bitmap.pixel_shader)
main.append(bg)

# --- FÕCÍM KIÍRÁSA ---
title = label.Label(
    terminalio.FONT,
    text="Blood pressure diagram 2025. december",
    color=0x000000,
    x=ML + 20,          # bal margóhoz igazítva
    y=MT - 8            # a keret fölé húzva
)
main.append(title)

# --- GÖRBÉK RAJZOLÁSA ---
graph = displayio.Group()

# --- SYS (piros) ---
for i in range(30):
    graph.append(Line(
        x_to_px(i+1), y_to_px(sys_values[i]),
        x_to_px(i+2), y_to_px(sys_values[i+1]),
        color=0xFF0000
    ))
    graph.append(Circle(x_to_px(i+1), y_to_px(sys_values[i]), 2, fill=0xFF0000))

graph.append(Circle(x_to_px(31), y_to_px(sys_values[30]), 2, fill=0xFF0000))


# --- DIA (fekete) ---
for i in range(30):
    graph.append(Line(
        x_to_px(i+1), y_to_px(dia_values[i]),
        x_to_px(i+2), y_to_px(dia_values[i+1]),
        color=0x000000
    ))
    graph.append(Circle(x_to_px(i+1), y_to_px(dia_values[i]), 2, fill=0x000000))

graph.append(Circle(x_to_px(31), y_to_px(dia_values[30]), 2, fill=0x000000))


main.append(graph)

# --- KIJELZÉS ---
display.root_group = main
display.refresh()

