"""
===============================================================
 E-PAPER DASHBOARD DEMÓPROGRAM
---------------------------------------------------------------
 Ez a program egy 296×128 pixeles, háromszínű (fekete–fehér–piros)
 e-paper kijelzőn jelenít meg egy statikus időjárási dashboardot.
 A cél: bemutatni a displayio rétegzett grafikus modelljét,
 ikonok, szövegek és háttérelemek kombinálásával.

 FŐBB FUNKCIÓK:
   • e-paper kijelző inicializálása (SPI + IL0373 driver)
   • háttér bitmap létrehozása
   • bal oldali információs blokk:
        - időjárás ikon (48×48 BMP)
        - városnév
        - aktuális idő (kiemelve pirossal)
        - dátum
   • jobb oldali információs blokk:
        - hőmérséklet ikon + érték
        - légnyomás ikon + érték
        - páratartalom ikon + érték

 FŐBB VÁLTOZÓK ÉS SZEREPÜK:
   spi              – SPI busz objektum
   epd_cs, epd_dc,
   epd_reset,
   epd_busy         – e-paper vezérlő lábkiosztása
   display_bus      – FourWire interfész a kijelzőhöz
   display          – IL0373 driver példány
   root             – a teljes képernyőt felépítő fő displayio.Group
   palette          – 3 színű paletta (fehér, fekete, piros)
   bg_bitmap        – 296×128 fehér háttér bitmap
   weather_bmp      – bal oldali nagy ikon (48×48)
   add_row()        – segédfüggvény: ikon + szöveg egy sorban

 KÉPELEMEK HIERARCHIÁJA (displayio struktúra):

   display.root_group
   └── root (Group)
       ├── bg (TileGrid: fehér háttér)
       ├── icon (TileGrid: bal oldali időjárás ikon)
       ├── city (Label: városnév)
       ├── time_label (Label: idő, piros)
       ├── date_label (Label: dátum)
       ├── [add_row(…)] hőmérséklet ikon + Label
       ├── [add_row(…)] légnyomás ikon + Label
       └── [add_row(…)] páratartalom ikon + Label

 MEGJEGYZÉS:
   A program statikus demó – a valós időjárási adatok lekérése
   és frissítése nincs implementálva, csak a grafikus elrendezés
   és megvalósításának szemléltetése a cél.

 TELEPÍTENDŐ KÖNYVTÁRAK (CIRCUITPYTHON /lib KÖNYVTÁRBA):
   adafruit_il0373.mpy
   adafruit_display_text/
       └── label.mpy

 TELEPÍTENDŐ IKON FÁJLOK (A GYÖKÉRKÖNYVTÁRBA):
   /icon1.bmp      – bal oldali nagy időjárás ikon (48×48 px)
   /temp.bmp       – hőmérséklet ikon (32×32 px)
   /press.bmp      – légnyomás ikon (32×32 px)
   /humidity.bmp   – páratartalom ikon (32×32 px)

 BEKÖTÉSI VÁZLAT:

   ESP32-C3        →     E-PAPER (IL0373 MODUL)
   --------------------------------------------
   3V3             →     VCC
   GND             →     GND
   board.IO6       →     SDI      (SPI MOSI)
   board.IO4       →     SCLK     (SPI órajel)
   board.IO3       →     CS       (Chip Select)
   board.IO2       →     DC       (Data/Command)
   board.IO1       →     RST      (Reset)
   board.IO0       →     BUSY     (Busy jel)
===============================================================
"""


import time
import board
import displayio
import terminalio
import fourwire
from adafruit_display_text import label
import adafruit_il0373

# ---------------------------------------------------------
# 1. KIJELZŐ INICIALIZÁLÁS
# ---------------------------------------------------------

displayio.release_displays()

spi = board.SPI()

epd_cs = board.IO3
epd_dc = board.IO2
epd_reset = board.IO1
epd_busy = board.IO0

display_bus = fourwire.FourWire(
    spi,
    command=epd_dc,
    chip_select=epd_cs,
    reset=epd_reset,
    baudrate=1_000_000,
)

display = adafruit_il0373.IL0373(
    display_bus,
    width=296,
    height=128,
    rotation=270,          # Landscape mód
    busy_pin=epd_busy,
    highlight_color=0xFF0000,  # Piros
)

# ---------------------------------------------------------
# 2. ROOT GROUP + FEHÉR HÁTTÉR
# ---------------------------------------------------------

root = displayio.Group()

# Háttér bitmap (fehér)
palette = displayio.Palette(3)
palette[0] = 0xFFFFFF   # fehér
palette[1] = 0x000000   # fekete
palette[2] = 0xFF0000   # piros

bg_bitmap = displayio.Bitmap(296, 128, 1)
bg_bitmap.fill(0)  # fehér
bg = displayio.TileGrid(bg_bitmap, pixel_shader=palette)
root.append(bg)

# ---------------------------------------------------------
# 3. BAL OLDAL: IDŐJÁRÁS IKON + HELY + IDŐ + DÁTUM
# ---------------------------------------------------------

weather_bmp = displayio.OnDiskBitmap("/icon1.bmp")
icon = displayio.TileGrid(weather_bmp, pixel_shader=weather_bmp.pixel_shader, x=4, y=4)
root.append(icon)

# Helynév
city = label.Label(
    terminalio.FONT,
    text="DEBRECEN",
    color=0x000000,
    scale=1,
    x=4,
    y=60,
)
root.append(city)

# Idő (nagy)
time_label = label.Label(
    terminalio.FONT,
    text="12:34",
    color=0xFF0000,
    scale=3,
    x=4,
    y=90,
)
root.append(time_label)

# Dátum
date_label = label.Label(
    terminalio.FONT,
    text="2026-01-13 Kedd",
    color=0x000000,
    scale=1,
    x=4,
    y=120,
)
root.append(date_label)

# ---------------------------------------------------------
# 4. JOBB OLDAL: HŐMÉRSÉKLET, LÉGNYOMÁS, PÁRATARTALOM
# ---------------------------------------------------------

def add_row(y, bmp_filename, text):
    """Egy sor: ikon + nagy szöveg."""
    bmp = displayio.OnDiskBitmap(bmp_filename)
    ic = displayio.TileGrid(bmp, pixel_shader=bmp.pixel_shader, x=110, y=y)
    root.append(ic)

    lbl = label.Label(
        terminalio.FONT,
        text=text,
        color=0x000000,
        scale=3,
        x=150,
        y=y + 16,
    )
    root.append(lbl)

add_row(10, "/temp.bmp", "12 °C")
add_row(50, "/press.bmp", "1013 hPa")
add_row(90, "/humidity.bmp", "68 %")

# ---------------------------------------------------------
# 5. MEGJELENÍTÉS
# ---------------------------------------------------------

display.root_group = root

while display.busy:
    pass

display.refresh()

print("Kész – a kijelző frissítve.")
